<?php
/**
 * Laporan Koperasi: Top Up & Transaksi (tanpa VIEW)
 * File: laporan.php
 * Disesuaikan untuk DB dump: `punp4156_koperasi_pp`.
 *
 * Sumber skema (dari dump):
 * - Database: punp4156_koperasi_pp
 * - Tabel: barang, santri, users, topup, transaksi, transaksi_items
 * - Kolom penting:
 *   - topup: (id_santri, id_user, nominal, status enum, created_at)
 *   - transaksi: (id_santri, id_user, total, tipe enum, status enum, created_at)
 *   - transaksi_items: (id_transaksi, id_barang, qty, harga_satuan, subtotal)
 *   - santri: nama; users: nama; barang: nama
 *
 * Catatan zona waktu:
 * - Dump set `SET time_zone = '+00:00'` (UTC). Agar laporan mengikuti WIB,
 *   query menggunakan CONVERT_TZ(...,'+00:00','+07:00').
 */

// ============ KONFIGURASI DB ============
$DB_HOST = 'localhost';
$DB_NAME = 'punp4156_koperasi_pp'; // disesuaikan dgn dump
$DB_USER = 'punp4156_koperasi_pp';                 // TODO: ganti user DB kamu
$DB_PASS = 'tugasakhir';                     // TODO: ganti password DB kamu
$DSN = "mysql:host={$DB_HOST};dbname={$DB_NAME};charset=utf8mb4";

// ============ KONEKSI DB ============
try {
  $pdo = new PDO($DSN, $DB_USER, $DB_PASS, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
} catch (Exception $e) {
  http_response_code(500);
  echo "Gagal koneksi database: " . htmlspecialchars($e->getMessage());
  exit;
}

// ============ UTIL: validasi dan parsing tanggal (YYYY-MM) ============
function ym_to_range($ym) {
  if (!preg_match('/^\d{4}\-(0[1-9]|1[0-2])$/', $ym)) return [null, null];
  [$y, $m] = explode('-', $ym);
  $start = sprintf('%04d-%02d-01 00:00:00', (int)$y, (int)$m);
  $y2 = (int)$y; $m2 = (int)$m + 1;
  if ($m2 === 13) { $m2 = 1; $y2++; }
  $end = sprintf('%04d-%02d-01 00:00:00', $y2, $m2);
  return [$start, $end];
}

// Ambil filter dari query string
$startYM = isset($_GET['start']) ? trim($_GET['start']) : '';
$endYM   = isset($_GET['end'])   ? trim($_GET['end'])   : '';

list($startAt, $tmp) = ym_to_range($startYM);
list($tmp2,   $endAt) = ym_to_range($endYM);
// Jika hanya start diisi tanpa end, end = start + 1 bulan
if ($startAt && !$endAt) {
  $dt = new DateTime($startAt);
  $dt->modify('+1 month');
  $endAt = $dt->format('Y-m-d H:i:s');
}

// ============ QUERY ============

// 1) Top Up per bulan
$sqlTopup = "
SELECT
  DATE_FORMAT(CONVERT_TZ(created_at,'+00:00','+07:00'), '%Y-%m') AS bulan,
  COUNT(*) AS jumlah_topup,
  SUM(nominal) AS total_topup
FROM topup
WHERE status='success'
  AND (:startAt IS NULL OR created_at >= :startAt)
  AND (:endAt   IS NULL OR created_at <  :endAt)
GROUP BY bulan
ORDER BY bulan;
";

// 2) Penjualan per bulan
$sqlPenjualan = "
SELECT
  DATE_FORMAT(CONVERT_TZ(created_at,'+00:00','+07:00'), '%Y-%m') AS bulan,
  COUNT(*) AS jumlah_transaksi,
  SUM(total) AS total_penjualan
FROM transaksi
WHERE status='success'
  AND tipe='pembelian'
  AND (:startAt IS NULL OR created_at >= :startAt)
  AND (:endAt   IS NULL OR created_at <  :endAt)
GROUP BY bulan
ORDER BY bulan;
";

// 3) Rekap gabungan per bulan
$sqlGabungan = "
SELECT
  bulan,
  SUM(CASE WHEN jenis='topup' THEN total ELSE 0 END) AS total_topup,
  SUM(CASE WHEN jenis='penjualan' THEN total ELSE 0 END) AS total_penjualan,
  SUM(total) AS grand_total
FROM (
  SELECT DATE_FORMAT(CONVERT_TZ(created_at,'+00:00','+07:00'),'%Y-%m') AS bulan,
         SUM(nominal) AS total, 'topup' AS jenis
  FROM topup
  WHERE status='success'
    AND (:startAt IS NULL OR created_at >= :startAt)
    AND (:endAt   IS NULL OR created_at <  :endAt)
  GROUP BY bulan

  UNION ALL

  SELECT DATE_FORMAT(CONVERT_TZ(created_at,'+00:00','+07:00'),'%Y-%m') AS bulan,
         SUM(total) AS total, 'penjualan' AS jenis
  FROM transaksi
  WHERE status='success' AND tipe='pembelian'
    AND (:startAt IS NULL OR created_at >= :startAt)
    AND (:endAt   IS NULL OR created_at <  :endAt)
  GROUP BY bulan
) x
GROUP BY bulan
ORDER BY bulan;
";

// 4) Detail transaksi
$sqlDetail = "
SELECT
  t.id AS id_transaksi,
  CONVERT_TZ(t.created_at,'+00:00','+07:00') AS waktu,
  s.nama AS santri,
  u.nama AS kasir,
  t.tipe, t.status,
  b.nama AS barang,
  ti.qty, ti.harga_satuan, ti.subtotal,
  t.total AS total_transaksi
FROM transaksi t
LEFT JOIN transaksi_items ti ON ti.id_transaksi = t.id
LEFT JOIN barang b           ON b.id = ti.id_barang
LEFT JOIN santri s           ON s.id = t.id_santri
LEFT JOIN users u            ON u.id = t.id_user
WHERE t.status='success' AND t.tipe='pembelian'
  AND (:startAt IS NULL OR t.created_at >= :startAt)
  AND (:endAt   IS NULL OR t.created_at <  :endAt)
ORDER BY t.created_at DESC, t.id
LIMIT 500;
";

// Eksekusi helper
function runQuery($pdo, $sql, $startAt, $endAt) {
  $stmt = $pdo->prepare($sql);
  $stmt->bindValue(':startAt', $startAt !== null ? $startAt : null, $startAt !== null ? PDO::PARAM_STR : PDO::PARAM_NULL);
  $stmt->bindValue(':endAt',   $endAt   !== null ? $endAt   : null, $endAt   !== null ? PDO::PARAM_STR : PDO::PARAM_NULL);
  $stmt->execute();
  return $stmt->fetchAll();
}

$topupRows     = runQuery($pdo, $sqlTopup,    $startAt, $endAt);
$jualRows      = runQuery($pdo, $sqlPenjualan,$startAt, $endAt);
$gabungRows    = runQuery($pdo, $sqlGabungan, $startAt, $endAt);
$detailRows    = runQuery($pdo, $sqlDetail,   $startAt, $endAt);

// Ringkasan kartu
function sum_field($rows, $field) {
  $sum = 0;
  foreach ($rows as $r) $sum += (float)($r[$field] ?? 0);
  return $sum;
}
$totalTopup     = sum_field($topupRows, 'total_topup');
$totalPenjualan = sum_field($jualRows, 'total_penjualan');
$grandTotal     = $totalTopup + $totalPenjualan;

// Data chart (sinkron dari gabungan)
$labels = []; $chartTopup = []; $chartJual = [];
foreach ($gabungRows as $r) {
  $labels[] = $r['bulan'];
  $chartTopup[] = (float)$r['total_topup'];
  $chartJual[]  = (float)$r['total_penjualan'];
}
function rupiah($n) { return 'Rp ' . number_format((float)$n, 0, ',', '.'); }

?>
<!doctype html>
<html lang="id">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1" />
  <title>Laporan Koperasi</title>
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <style>
    :root { --bg:#0f172a; --muted:#94a3b8; --text:#e5e7eb; }
    *{box-sizing:border-box} body{margin:0;font-family:Inter,system-ui,Arial;background:#0b1220;color:var(--text)}
    .wrap{max-width:1180px;margin:32px auto;padding:0 16px}
    h1{font-size:28px;margin:0 0 16px}
    .sub{color:var(--muted);margin-bottom:24px}
    .panel{background:#0f172a;border:1px solid #1f2937;border-radius:16px;padding:16px;margin-bottom:16px}
    .grid{display:grid;gap:16px}
    .g-3{grid-template-columns:repeat(3,minmax(0,1fr))}
    .card{background:linear-gradient(180deg,#0f172a,#0b1220);border:1px solid #1f2937;border-radius:16px;padding:16px}
    .card h3{margin:0 0 8px;font-size:14px;color:var(--muted);font-weight:600;letter-spacing:.02em;text-transform:uppercase}
    .card .v{font-size:24px;font-weight:700}
    .filter{display:flex;gap:8px;align-items:center;flex-wrap:wrap}
    input, button, select{background:#0b1220;border:1px solid #1f2937;color:var(--text);padding:10px 12px;border-radius:12px}
    button{cursor:pointer}
    table{width:100%;border-collapse:collapse}
    th, td{padding:10px;border-bottom:1px solid #1f2937;text-align:left}
    th{color:var(--muted);font-weight:600}
    .section-title{display:flex;justify-content:space-between;align-items:center;margin:24px 0 8px}
    .muted{color:var(--muted)}
    .chartbox{background:#0f172a;border:1px solid #1f2937;border-radius:16px;padding:16px}
    .footer{color:#64748b;font-size:12px;margin-top:24px}
    a{color:#7dd3fc;text-decoration:none}
  </style>
</head>
<body>
  <div class="wrap">
    <h1>Laporan Koperasi</h1>
    <div class="panel filter">
      <form method="get" class="filter">
        <label>Mulai (YYYY-MM): <input type="month" name="start" value="<?= htmlspecialchars($startYM ?: '') ?>"></label>
        <label>Selesai (YYYY-MM): <input type="month" name="end" value="<?= htmlspecialchars($endYM ?: '') ?>"></label>
        <button type="submit">Terapkan</button>
        <a href="?">Reset</a>
      </form>
    </div>

    <div class="grid g-3">
      <div class="card"><h3>Total Top Up</h3><div class="v"><?= rupiah($totalTopup) ?></div></div>
      <div class="card"><h3>Total Penjualan</h3><div class="v"><?= rupiah($totalPenjualan) ?></div></div>
      <div class="card"><h3>Grand Total</h3><div class="v"><?= rupiah($grandTotal) ?></div></div>
    </div>

    <div class="section-title">
      <h2>Grafik Per Bulan</h2><div class="muted">Top Up vs Penjualan</div>
    </div>
    <div class="chartbox"><canvas id="chartMonthly" height="90"></canvas></div>

    <div class="section-title"><h2>Top Up per Bulan</h2></div>
    <div class="panel">
      <table>
        <thead><tr><th>Bulan</th><th class="muted">Jumlah</th><th class="muted">Total Top Up</th></tr></thead>
        <tbody>
          <?php foreach ($topupRows as $r): ?>
            <tr><td><?= htmlspecialchars($r['bulan']) ?></td><td><?= (int)$r['jumlah_topup'] ?></td><td><?= rupiah($r['total_topup']) ?></td></tr>
          <?php endforeach; if (!$topupRows): ?>
            <tr><td colspan="3" class="muted">Tidak ada data.</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <div class="section-title"><h2>Penjualan per Bulan</h2></div>
    <div class="panel">
      <table>
        <thead><tr><th>Bulan</th><th class="muted">Jumlah Transaksi</th><th class="muted">Total Penjualan</th></tr></thead>
        <tbody>
          <?php foreach ($jualRows as $r): ?>
            <tr><td><?= htmlspecialchars($r['bulan']) ?></td><td><?= (int)$r['jumlah_transaksi'] ?></td><td><?= rupiah($r['total_penjualan']) ?></td></tr>
          <?php endforeach; if (!$jualRows): ?>
            <tr><td colspan="3" class="muted">Tidak ada data.</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <div class="section-title"><h2>Rekap Gabungan per Bulan</h2></div>
    <div class="panel">
      <table>
        <thead><tr><th>Bulan</th><th class="muted">Total Top Up</th><th class="muted">Total Penjualan</th><th class="muted">Grand Total</th></tr></thead>
        <tbody>
          <?php foreach ($gabungRows as $r): ?>
            <tr>
              <td><?= htmlspecialchars($r['bulan']) ?></td>
              <td><?= rupiah($r['total_topup']) ?></td>
              <td><?= rupiah($r['total_penjualan']) ?></td>
              <td><?= rupiah($r['grand_total']) ?></td>
            </tr>
          <?php endforeach; if (!$gabungRows): ?>
            <tr><td colspan="4" class="muted">Tidak ada data.</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <div class="section-title"><h2>Detail Transaksi</h2><div class="muted">Maks 500 baris terbaru</div></div>
    <div class="panel">
      <table>
        <thead>
          <tr>
            <th>ID</th><th>Waktu</th><th>Santri</th><th>Kasir</th><th>Tipe</th><th>Status</th>
            <th>Barang</th><th>Qty</th><th>Harga</th><th>Subtotal</th><th>Total Transaksi</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($detailRows as $r): ?>
            <tr>
              <td><?= (int)$r['id_transaksi'] ?></td>
              <td><?= htmlspecialchars($r['waktu']) ?></td>
              <td><?= htmlspecialchars($r['santri']) ?></td>
              <td><?= htmlspecialchars($r['kasir']) ?></td>
              <td><?= htmlspecialchars($r['tipe']) ?></td>
              <td><?= htmlspecialchars($r['status']) ?></td>
              <td><?= htmlspecialchars($r['barang']) ?></td>
              <td><?= (int)$r['qty'] ?></td>
              <td><?= rupiah($r['harga_satuan']) ?></td>
              <td><?= rupiah($r['subtotal']) ?></td>
              <td><?= rupiah($r['total_transaksi']) ?></td>
            </tr>
          <?php endforeach; if (!$detailRows): ?>
            <tr><td colspan="11" class="muted">Tidak ada data.</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <div class="footer">
      <p>Halaman ini adalah <strong>halaman laporan dari koperasi</strong>. 
    </div>
  </div>

<script>
(function() {
  const ctx = document.getElementById('chartMonthly');
  if (!ctx) return;
  const labels = <?php echo json_encode($labels); ?>;
  const dataTopup = <?php echo json_encode($chartTopup); ?>;
  const dataJual  = <?php echo json_encode($chartJual); ?>;
  new Chart(ctx, {
    type: 'bar',
    data: {
      labels: labels,
      datasets: [
        { label: 'Top Up',    data: dataTopup, borderWidth: 1 },
        { label: 'Penjualan', data: dataJual,  borderWidth: 1 }
      ]
    },
    options: {
      responsive: true,
      scales: { y: { beginAtZero: true } },
      plugins: {
        legend: { position: 'top' },
        tooltip: { mode: 'index', intersect: false }
      }
    }
  });
})();
</script>
</body>
</html>
