<?php
// ====== KONFIGURASI DASAR ======
$DB_HOST = 'localhost';
$DB_NAME = 'punp4156_koperasi_pp';
$DB_USER = 'punp4156_koperasi_pp';
$DB_PASS = 'tugasakhir';

// Batas minimal & maksimal top up (sesuaikan kebijakan)
$TOPUP_MIN = 5000;     // Rp 5.000
$TOPUP_MAX = 1000000;  // Rp 1.000.000

// ====== KONEKSI DB ======
$pdo = null;
try {
  $pdo = new PDO(
    "mysql:host={$DB_HOST};dbname={$DB_NAME};charset=utf8mb4",
    $DB_USER,
    $DB_PASS,
    [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
  );
} catch (Exception $e) {
  http_response_code(500);
  echo "<pre>Gagal konek DB: ".$e->getMessage()."</pre>";
  exit;
}

// ====== HELPER JSON ======
function jsonOut($arr, $code = 200) {
  http_response_code($code);
  // Anti-cache: supaya data riwayat selalu fresh
  header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
  header('Pragma: no-cache');
  header('Content-Type: application/json');
  echo json_encode($arr);
  exit;
}

// ====== ROUTING AJAX (DALAM SATU FILE) ======
$ajax = $_GET['ajax'] ?? null;

if ($ajax === 'latest') {
  // Ambil UID terakhir dari rfid_latest (id=1)
  $uid = $pdo->query("SELECT uid FROM rfid_latest WHERE id=1")->fetchColumn();

  $santri = null;
  if ($uid) {
    $st = $pdo->prepare("SELECT id, nama, saldo, aktif, rfid_uid FROM santri WHERE rfid_uid = ? LIMIT 1");
    $st->execute([$uid]);
    $santri = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  }

  jsonOut(['uid' => $uid, 'santri' => $santri, 'limits' => ['min' => $TOPUP_MIN, 'max' => $TOPUP_MAX]]);
}

// Riwayat global semua santri (permanen)
if ($ajax === 'history_all') {
  $limit = 100; // ubah kalau mau lebih banyak
  $sql = "SELECT t.id, t.nominal, t.status, t.created_at,
                 s.nama AS nama_santri, s.rfid_uid AS uid
          FROM topup t
          LEFT JOIN santri s ON s.id = t.id_santri
          ORDER BY t.created_at DESC, t.id DESC
          LIMIT {$limit}";
  $rows = $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC);
  jsonOut($rows);
}

if ($ajax === 'topup' && $_SERVER['REQUEST_METHOD'] === 'POST') {
  $input = json_decode(file_get_contents('php://input'), true);
  $uid   = trim($input['uid'] ?? '');
  // Normalisasi UID
  $uid = strtoupper(trim($uid));
  $nom   = (int)($input['nominal'] ?? 0);
  $idUser= (int)($input['id_user'] ?? 2); // opsional: id kasir yang login

  if ($nom < $TOPUP_MIN) jsonOut(['error' => 'Minimal top up Rp '.number_format($TOPUP_MIN,0,',','.')], 422);
  if ($nom > $TOPUP_MAX) jsonOut(['error' => 'Maksimal top up Rp '.number_format($TOPUP_MAX,0,',','.')], 422);

  // Pastikan benar-benar ada kartu yang discan saat ini dan cocok dengan UID request
  $uidLatest = $pdo->query("SELECT uid FROM rfid_latest WHERE id=1")->fetchColumn();
  $uidLatest = strtoupper(trim((string)$uidLatest));

  if (!$uidLatest || $uidLatest !== $uid) {
    usleep(100000); // 0.1s
    $uidLatest = $pdo->query("SELECT uid FROM rfid_latest WHERE id=1")->fetchColumn();
    $uidLatest = strtoupper(trim((string)$uidLatest));
    if (!$uidLatest || $uidLatest !== $uid) {
      jsonOut(['error' => 'Silakan scan kartu terlebih dahulu'], 422);
    }
  }

  // Cari santri dari UID
  $st = $pdo->prepare("SELECT id, saldo, aktif FROM santri WHERE rfid_uid=? LIMIT 1");
  $st->execute([$uid]);
  $santri = $st->fetch(PDO::FETCH_ASSOC);
  if (!$santri) jsonOut(['error' => 'Kartu belum terdaftar'], 404);
  if ((int)$santri['aktif'] !== 1) jsonOut(['error' => 'Akun santri tidak aktif'], 422);

  try {
    $pdo->beginTransaction();

    // Catat ke tabel topup
    $ins = $pdo->prepare("INSERT INTO topup (id_santri, id_user, nominal, status) VALUES (?, ?, ?, 'success')");
    $ins->execute([$santri['id'], $idUser, $nom]);
    $idTrans = (int)$pdo->lastInsertId();

    // Update saldo
    $upd = $pdo->prepare("UPDATE santri SET saldo = saldo + ? WHERE id = ?");
    $upd->execute([$nom, $santri['id']]);

    $pdo->commit();

    $saldoBaru = $pdo->query("SELECT saldo FROM santri WHERE id=".(int)$santri['id'])->fetchColumn();
    jsonOut(['ok' => true, 'id_transaksi' => $idTrans, 'saldo_baru' => (int)$saldoBaru]);
  } catch (Exception $e) {
    $pdo->rollBack();
    jsonOut(['error' => 'Gagal top up: '.$e->getMessage()], 500);
  }
}

// ====== HALAMAN UI ======
?>
<!doctype html>
<html lang="id">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Menu Top Up</title>
  <style>
    :root{
      --radius:14px; --surface:#ffffff; --line:#e5e7eb;
      --text:#111827; --muted:#6b7280; --primary:#1363DF; --primary-fg:#fff;
      --bg:#f8fafc; --ok:#047857; --err:#b91c1c;
    }
    *{ box-sizing:border-box; }
    body{ background:var(--bg); color:var(--text); font-family: system-ui, -apple-system, Segoe UI, Roboto, Arial; }
    .container{ max-width: 960px; margin: 28px auto; padding: 0 14px; }
    .hero{
      border-radius: 18px; padding: 28px;
      background: radial-gradient(1200px 600px at 0% 0%, #e0efff 0%, #f8fbff 40%, #ffffff 100%);
      border: 1px solid var(--line);
      box-shadow: 0 10px 30px rgba(19,99,223,.06);
      margin-bottom: 18px;
    }
    h1{ margin:0 0 6px; font-size: 28px; }
    p.lead{ margin:0; color:var(--muted); }
    .row{ display:flex; gap:14px; flex-wrap:wrap; align-items:center; }
    .btn{
      display:inline-flex; align-items:center; gap:10px;
      padding:12px 16px; border-radius: 12px; border:1px solid transparent;
      cursor:pointer; font-weight:700; letter-spacing:.2px;
      transition:transform .05s ease, box-shadow .2s ease, background .2s ease;
      user-select:none;
    }
    .btn:active{ transform: translateY(1px); }
    .btn-primary{ background:var(--primary); color:var(--primary-fg); box-shadow: 0 4px 18px rgba(19,99,223,.25); }
    .btn-secondary{ background:#f3f4f6; color:#111827; border-color:var(--line); }
    .card{
      background:var(--surface); border:1px solid var(--line);
      border-radius: var(--radius); padding:18px; margin-bottom:16px;
      box-shadow: 0 6px 18px rgba(0,0,0,.04);
    }
    .muted{ color:var(--muted); }
    .ok{ color:var(--ok); }
    .err{ color:var(--err); }
    .mono{ font-family: ui-monospace, SFMono-Regular, Menlo, Consolas, "Liberation Mono", monospace; }
    label{ display:block; font-size:.95rem; margin-bottom:6px; color:#374151; }
    input[type=number]{ width:100%; padding:12px 14px; border:1px solid var(--line); border-radius:12px; font-size:1rem; background:#fff; }
    table{ width:100%; border-collapse: collapse; }
    th,td{ padding:10px 12px; border-bottom:1px solid #f1f5f9; text-align:left; }
    .right{ text-align:right; }
    .grid{ display:grid; grid-template-columns: repeat(3, minmax(0,1fr)); gap:12px; }
    @media (max-width:720px){ .grid{ grid-template-columns: 1fr; } }
    .pill{ display:inline-block; padding:6px 10px; border-radius:999px; background:#f1f5f9; color:#0f172a; font-weight:600; font-size:.85rem; }
  </style>
</head>
<body>
  <div class="container">
    <!-- HERO: Scan -->
    <div class="hero">
      <h1>Top Up Saldo</h1>
      <p class="lead">Klik <b>Scan Kartu</b> lalu tempelkan kartu di reader.</p>
      <div class="row" style="margin-top:14px;">
        <button id="btnScan" type="button" class="btn btn-primary">🔄 Scan Kartu</button>
        <span id="scanStatus" class="muted">Belum ada kartu yang discan.</span>
      </div>
    </div>

    <!-- DATA SANTRI (muncul setelah scan) -->
    <div id="santriCard" class="card" style="display:none;">
      <div class="grid">
        <div>
          <div class="muted">UID Kartu</div>
          <div class="mono" id="uid">—</div>
        </div>
        <div>
          <div class="muted">Nama Santri</div>
          <div id="nama">—</div>
        </div>
        <div>
          <div class="muted">Saldo Saat Ini</div>
          <div id="saldo">—</div>
        </div>
      </div>
      <div class="muted" style="margin-top:8px;">Terakhir diperbarui: <span id="seen">—</span></div>
    </div>

    <!-- FORM TOP UP -->
    <div class="card">
      <form id="formTopup">
        <div class="row">
          <div style="flex: 1 1 320px;">
            <label for="nominal">Nominal Top Up</label>
            <input id="nominal" type="number" min="<?= (int)$TOPUP_MIN ?>" max="<?= (int)$TOPUP_MAX ?>" step="500" required />
            <small class="muted">Min Rp <?= number_format($TOPUP_MIN,0,',','.') ?>, Max Rp <?= number_format($TOPUP_MAX,0,',','.') ?></small>
          </div>
          <div style="align-self:end;">
            <button class="btn btn-secondary" type="button" id="btnClear">Bersihkan</button>
            <button class="btn btn-primary" type="submit" id="btnSubmit">Proses Top Up</button>
          </div>
        </div>
        <div id="msg" style="margin-top:10px;"></div>
        <div id="scanNotice" class="muted" style="margin-top:8px;">Silakan <b>scan kartu</b> terlebih dahulu untuk top up.</div>
      </form>
    </div>

    <!-- RIWAYAT GLOBAL -->
    <div class="card">
      <div class="row" style="justify-content:space-between; align-items:center;">
        <h3 style="margin:0;">Riwayat Top Up</h3>
        <span class="pill" id="riwayatAllCount">0 Transaksi</span>
      </div>
      <div style="overflow:auto; margin-top:10px;">
        <table>
          <thead>
            <tr>
              <th>Tanggal</th>
              <th>ID Transaksi</th>
              <th>UID</th>
              <th>Nama Santri</th>
              <th class="right">Nominal</th>
              <th>Status</th>
            </tr>
          </thead>
          <tbody id="historyAll">
            <tr><td colspan="6" class="muted">Belum ada data</td></tr>
          </tbody>
        </table>
      </div>
    </div>
  </div>

<script>
const el = (id)=>document.getElementById(id);
const fmtRp = n => new Intl.NumberFormat('id-ID').format(n);

let currentUID = null;
let currentSantri = null;
let limits = {min: <?= (int)$TOPUP_MIN ?>, max: <?= (int)$TOPUP_MAX ?>};
let scanning = false;

const formTopup = document.getElementById('formTopup');
const inputNominal = document.getElementById('nominal');
const btnSubmit = document.getElementById('btnSubmit');
const btnClear = document.getElementById('btnClear');
const btnScan = document.getElementById('btnScan');
const scanStatus = document.getElementById('scanStatus');
const santriCard = document.getElementById('santriCard');
const scanNotice = document.getElementById('scanNotice');

function setFormEnabled(enabled) {
  inputNominal.disabled = !enabled;
  btnSubmit.disabled = !enabled;
  scanNotice.style.display = enabled ? 'none' : 'block';
}
setFormEnabled(false); // awalnya nonaktif

async function fetchLatest(){
  if (!scanning) return; // hanya setelah klik scan
  try{
    const res = await fetch('?ajax=latest');
    const data = await res.json();
    const uid = data?.uid || null;
    const s = data?.santri || null;
    limits = data?.limits || limits;

    if (s && s.id) {
      scanStatus.innerHTML = '✅ Kartu terdeteksi.';
      santriCard.style.display = 'block';
      el('uid').textContent = uid || '—';
      el('nama').textContent = s?.nama || '—';
      el('saldo').textContent = s ? 'Rp ' + fmtRp(s.saldo) : '—';
      el('seen').textContent = new Date().toLocaleString('id-ID');
      setFormEnabled(true);
    } else {
      scanStatus.innerHTML = '❌ Kartu tidak terdeteksi. Coba tempelkan lagi lalu klik <b>Scan Kartu</b>.';
      santriCard.style.display = 'none';
      setFormEnabled(false);
      inputNominal.value = '';
    }

    // Simpan state terbaru
    currentUID = uid;
    currentSantri = s || null;
  }catch(e){
    scanStatus.innerHTML = '⚠️ Gagal memindai. Periksa koneksi.';
  }
}

// Riwayat global: selalu dimuat saat halaman dibuka & setelah top up
async function loadHistoryAll() {
  try {
    const res = await fetch('?ajax=history_all', { cache: 'no-store' });
    const rows = await res.json();

    const tbody = el('historyAll');
    tbody.innerHTML = '';

    if (!Array.isArray(rows) || rows.length === 0) {
      tbody.innerHTML = '<tr><td colspan="6" class="muted">Belum ada data</td></tr>';
      el('riwayatAllCount').textContent = '0 Transaksi';
      return;
    }

    el('riwayatAllCount').textContent = rows.length + ' Transaksi';

    rows.forEach(r => {
      const tr = document.createElement('tr');
      tr.innerHTML = `
        <td>${new Date(r.created_at).toLocaleString('id-ID')}</td>
        <td class="mono">#${r.id}</td>
        <td class="mono">${r.uid ?? '—'}</td>
        <td>${r.nama_santri ?? '—'}</td>
        <td class="mono right">Rp ${fmtRp(r.nominal)}</td>
        <td>${r.status}</td>
      `;
      tbody.appendChild(tr);
    });
  } catch (e) {
    const tbody = el('historyAll');
    tbody.innerHTML = '<tr><td colspan="6" class="muted">Gagal memuat riwayat</td></tr>';
    el('riwayatAllCount').textContent = '—';
  }
}

// Muat riwayat global ketika halaman selesai dimuat
document.addEventListener('DOMContentLoaded', () => {
  loadHistoryAll();
});

// Tombol Bersihkan
btnClear.addEventListener('click', () => {
  el('nominal').value = '';
  el('msg').textContent = '';
  loadHistoryAll();
});

// Submit topup
formTopup.addEventListener('submit', async (ev)=>{
  ev.preventDefault();
  if (btnSubmit.disabled) return;

  el('msg').textContent = '';

  if (!currentUID || !currentSantri?.id){
    el('msg').innerHTML = '<span class="err">Kartu belum terdeteksi / tidak terdaftar.</span>';
    return;
  }

  const nominal = parseInt(el('nominal').value||'0',10);
  if (nominal < limits.min) { el('msg').innerHTML = '<span class="err">Minimal top up Rp '+fmtRp(limits.min)+'</span>'; return; }
  if (nominal > limits.max) { el('msg').innerHTML = '<span class="err">Maksimal top up Rp '+fmtRp(limits.max)+'</span>'; return; }

  try{
    const res = await fetch('?ajax=topup', {
      method: 'POST',
      headers: {'Content-Type':'application/json'},
      body: JSON.stringify({ uid: currentUID, nominal, id_user: 2 })
    });
    const out = await res.json();
    if (!res.ok) throw new Error(out.error || 'Gagal top up');

    el('msg').innerHTML = '<span class="ok">Top up berhasil. Saldo baru: Rp '+fmtRp(out.saldo_baru)+'</span>';
    el('nominal').value = '';

    // refresh data kartu & riwayat global
    await fetchLatest();
    await loadHistoryAll();
  }catch(e){
    el('msg').innerHTML = '<span class="err">'+e.message+'</span>';
  }
});

// Tombol scan → sekali baca
btnScan.addEventListener('click', async ()=>{
  scanning = true;
  scanStatus.textContent = 'Memindai... tempelkan kartu sekarang.';
  await fetchLatest();
  scanning = false;
});
</script>
</body>
</html>
